<?php
/**
 * Plugin generic functions file
 *
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */

if ( !defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Update default settings
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_default_settings() {

	global $wpls_pro_options;

	$wpls_pro_options = array(
		'tooltip_theme'		=> 'punk',
		'tooltip_animation'	=> 'grow',
		'tooltip_behavior'	=> 'hover',
		'tooltip_arrow'		=> 'true',
		'tooltip_delay'		=> 300,
		'tooltip_distance'	=> 6,
		'tooltip_maxwidth'	=> '',
		'tooltip_minwidth'	=> '',
		'custom_css' 		=> '',
	);

	$default_options = apply_filters('wpls_pro_options_default_values', $wpls_pro_options );

	// Update default options
	update_option( 'wpls_pro_options', $default_options );

	// Overwrite global variable when option is update
	$wpls_pro_options = wpls_pro_get_settings();
}

/**
 * Get Settings From Option Page
 * 
 * Handles to return all settings value
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_get_settings() {

	$options = get_option('wpls_pro_options');

	$settings = is_array($options) 	? $options : array();

	return $settings;
}

/**
 * Get an option
 * Looks to see if the specified setting exists, returns default if not
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_get_option( $key = '', $default = false ) {
	global $wpls_pro_options;

	$value = ! empty( $wpls_pro_options[ $key ] ) ? $wpls_pro_options[ $key ] : $default;
	$value = apply_filters( 'wpls_pro_get_option', $value, $key, $default );
	return apply_filters( 'wpls_pro_get_option_' . $key, $value, $key, $default );
}

/**
 * Sanitize URL
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.3.7
 */
function wpls_pro_clean_url( $url ) {
	return esc_url_raw( trim($url) );
}

/**
 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
 * Non-scalar values are ignored.
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0
 */
function wpls_pro_clean( $var ) {
	if ( is_array( $var ) ) {
		return array_map( 'wpls_pro_clean', $var );
	} else {
		$data = is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
		return wp_unslash($data);
	}
}

/**
 * Sanitize text area value
 *
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.3.7
 */
function wpls_pro_clean_html($data = array(), $flag = false) {

    if($flag != true) {
        $data = wpls_pro_nohtml_kses($data);
    }
    $data = stripslashes_deep($data);
    return $data;
}

/**
 * Strip Html Tags 
 * 
 * It will sanitize text input (strip html tags, and escape characters)
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_nohtml_kses($data = array()) {

	if ( is_array($data) ) {

		$data = array_map('wpls_pro_nohtml_kses', $data);

	} elseif ( is_string( $data ) ) {
		$data = trim( $data );
		$data = wp_filter_nohtml_kses($data);
	}

	return $data;
}

/**
 * Escape Tags & Slashes
 *
 * Handles escapping the slashes and tags
 *
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_esc_attr($data) {
	return esc_attr( stripslashes($data) );
}

/**
 * Sanitize Multiple HTML class
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0
 */
function wpls_pro_sanitize_html_classes($classes, $sep = " ") {
	$return = "";

	if( !is_array($classes) ) {
		$classes = explode($sep, $classes);
	}

	if( !empty($classes) ) {
		foreach($classes as $class){
			$return .= sanitize_html_class($class) . " ";
		}
		$return = trim( $return );
	}

	return $return;
}

/**
 * Function to add array after specific key
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_add_array(&$array, $value, $index, $from_last = false) {

	if( is_array($array) && is_array($value) ) {

		if( $from_last ) {
			$total_count    = count($array);
			$index          = (!empty($total_count) && ($total_count > $index)) ? ($total_count-$index): $index;
		}

		$split_arr  = array_splice($array, max(0, $index));
		$array      = array_merge( $array, $value, $split_arr);
	}

	return $array;
}

/**
 * get featured image of logo post
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_get_logo_image( $post_id = '', $size = 'full' ) {

	$prefix = WPLS_META_PREFIX; // Metabox prefix

	// Taking external image
	$image = get_post_meta( $post_id, $prefix.'logo_url', true );

	// If external image is blank then take featured image
	if( empty($image) ) {
		$image = wp_get_attachment_image_src( get_post_thumbnail_id( $post_id ), $size );

		if( !empty($image) ) {
			$image = isset($image[0]) ? $image[0] : '';
		}
	}

	return $image;
}

/**
 * Function to get post excerpt
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_get_post_excerpt( $post_id = null, $content = '', $word_length = '55', $more = '...' ) {

	$has_excerpt 	= false;
	$word_length 	= !empty($word_length) ? $word_length : '55';

	// If post id is passed
	if( !empty($post_id) ) {
		if (has_excerpt($post_id)) {

			$has_excerpt 	= true;
			$content 		= get_the_excerpt();

		} else {
			$content = !empty($content) ? $content : get_the_content();
		}
	}

	if( !empty($content) && (!$has_excerpt) ) {
		$content = strip_shortcodes( $content ); // Strip shortcodes
		$content = wp_trim_words( $content, $word_length, $more );
	}

	return $content;
}

/**
 * Function to get plugin image sizes array
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_get_unique() {
	static $unique = 0;
	$unique++;

	return $unique;
}

/**
 * Pagination function for grid
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0
 */
function wpls_pro_pagination( $args = array() ) {

	$big = 999999999; // need an unlikely integer

	$paging = apply_filters('wpls_pro_post_paging_args', array(
					'base' 		=> str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
					'format' 	=> '?paged=%#%',
					'current' 	=> max( 1, $args['paged'] ),
					'total'		=> $args['total'],
					'prev_next'	=> true,
					'prev_text'	=> '&laquo; '.__('Previous', 'logoshowcase'),
					'next_text'	=> __('Next', 'logoshowcase').' &raquo;',
				));

	return paginate_links($paging);
}

/**
 * Function to get `logoshowcase` shortcode designs
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_logo_designs() {
	$design_arr = array(
						'design-1' 		=> __('Design 1', 'logoshowcase'),
						'design-2' 		=> __('Design 2', 'logoshowcase'),
						'design-3' 		=> __('Design 3', 'logoshowcase'),
						'design-4' 		=> __('Design 4', 'logoshowcase'),
						'design-5' 		=> __('Design 5', 'logoshowcase'),
						'design-6' 		=> __('Design 6', 'logoshowcase'),
						'design-7' 		=> __('Design 7', 'logoshowcase'),
						'design-8' 		=> __('Design 8', 'logoshowcase'),
						'design-9' 		=> __('Design 9', 'logoshowcase'),
						'design-10' 	=> __('Design 10', 'logoshowcase'),
						'design-11' 	=> __('Design 11', 'logoshowcase'),
						'design-12' 	=> __('Design 12', 'logoshowcase'),
						'design-13' 	=> __('Design 13', 'logoshowcase'),
						'design-14' 	=> __('Design 14', 'logoshowcase'),
						'design-15' 	=> __('Design 15', 'logoshowcase'),
						'design-16' 	=> __('Design 16', 'logoshowcase'),
		);
	return apply_filters('wpls_pro_logo_designs', $design_arr );
}

/**
 * Function to get supported animations
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */
function wpls_pro_animations() {
	$animations = array(
			'flash' 		=> __('Flash', 'logoshowcase'),
			'pulse'			=> __('Pulse', 'logoshowcase'),
			'rubberBand' 	=> __('Rubber Band', 'logoshowcase'),
			'headShake'		=> __('Head Shake', 'logoshowcase'),
			'swing'			=> __('Swing', 'logoshowcase'),
			'tada'			=> __('Tada', 'logoshowcase'),
			'wobble'		=> __('Wobble', 'logoshowcase'),
			'jello'			=> __('jello', 'logoshowcase'),
			'bounceIn'		=> __('Bounce In', 'logoshowcase'),
			'fadeIn'		=> __('Fade In', 'logoshowcase'),
			'fadeOut'		=> __('Fade Out', 'logoshowcase'),
			'lightSpeedOut'	=> __('Light Speed Out', 'logoshowcase'),
			'rotateIn'		=> __('Rotate In', 'logoshowcase'),
		);
	return apply_filters('wpls_pro_animations', $animations );
}

/**
 * Function to get shortocdes registered in plugin
 * 
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.3.4
 */
function wpls_pro_registered_shortcodes() {
	$shortcodes = array(
					'logoshowcase'	=> __('Logo Showcase Slider', 'logoshowcase'),
					'logo_grid' 	=> __('Logo Showcase Grid', 'logoshowcase'),
					'logo_filter' 	=> __('Logo Showcase Filter', 'logoshowcase'),
				);
	return apply_filters('wpls_pro_registered_shortcodes', (array)$shortcodes );
}