<?php
/**
 * Script Class
 *
 * Handles the script and style functionality of plugin
 *
 * @package WP Logo Showcase Responsive Slider Pro
 * @since 1.0.0
 */

if ( !defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Wpls_Pro_Admin {
	
	function __construct() {

		// Action to add metabox
		add_action( 'add_meta_boxes', array($this, 'wpls_pro_post_sett_metabox'), 10, 2 );

		// Action to save metabox value
		add_action( 'save_post', array($this, 'wpls_pro_save_meta_box_data') );

		// Action to register admin menu
		add_action( 'admin_menu', array($this, 'wpls_pro_register_menu'), 9 );

		// Shortocde Preview
		add_action( 'current_screen', array($this, 'wpls_pro_generate_preview_screen') );

		// Filter to add row action in category table
		add_filter( WPLS_PRO_CAT.'_row_actions', array($this, 'wpls_pro_add_tax_row_data'), 10, 2 );

		// Action to register plugin settings
		add_action ( 'admin_init', array($this, 'wpls_pro_admin_processes') );

		// Action to add custom column to Logo listing
		add_filter( 'manage_'.WPLS_PRO_POST_TYPE.'_posts_columns', array($this, 'wpls_pro_posts_columns') );

		// Action to add custom column data to Logo listing
		add_action('manage_'.WPLS_PRO_POST_TYPE.'_posts_custom_column', array($this, 'wpls_pro_post_columns_data'), 10, 2);
		
		// Action to add category filter dropdown
		add_action( 'restrict_manage_posts', array($this, 'wpls_pro_add_post_filters'), 50 );
		
		// Action to add sorting link at Logo listing page
		add_filter( 'views_edit-'.WPLS_PRO_POST_TYPE, array($this, 'wpls_pro_sorting_link') );
		
		// Action to add `Save Order` button
		add_action( 'restrict_manage_posts', array($this, 'wpls_pro_restrict_manage_posts') );
		
		// Ajax call to update option
		add_action( 'wp_ajax_wpls_pro_update_post_order', array($this, 'wpls_pro_update_post_order'));
		add_action( 'wp_ajax_nopriv_wpls_pro_update_post_order',array( $this, 'wpls_pro_update_post_order'));

		// Filter to add plugin links
		add_filter( 'plugin_row_meta', array( $this, 'wpls_pro_plugin_row_meta' ), 10, 2 );
	}

	/**
	 * Post Settings Metabox
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_post_sett_metabox( $post_type, $post ) {
		add_meta_box( 'wpls-pro-post-sett', __('WP Logo Showcase Responsive Slider Pro - Settings', 'logoshowcase'), array($this, 'wpls_pro_post_sett_mb_content'), WPLS_PRO_POST_TYPE, 'normal', 'high' );
	}

	/**
	 * Function to handle 'Add Link URL' metabox HTML
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_post_sett_mb_content( $post ) {
		include_once( WPLS_PRO_DIR .'/includes/admin/metabox/wpls-post-sett-metabox.php');
	}

	/**
	 * Function to save metabox values
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_save_meta_box_data( $post_id ){

		global $post_type;
		
		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )                	// Check Autosave
		|| ( ! isset( $_POST['post_ID'] ) || $post_id != $_POST['post_ID'] )  	// Check Revision
		|| ( $post_type !=  WPLS_PRO_POST_TYPE ) )              				// Check if current post type is supported.
		{
		  return $post_id;
		}

		$prefix = WPLS_META_PREFIX; // Taking metabox prefix

		// Taking data
		$logo_url 	= isset($_POST[$prefix.'logo_url'])		? wpls_pro_clean_url( $_POST[$prefix.'logo_url'] ) 	: '';
		$logo_link 	= isset($_POST[$prefix.'logo_link']) 	? wpls_pro_clean_url( $_POST[$prefix.'logo_link'] ) 	: '';

		// Updating Post Meta
		update_post_meta( $post_id, $prefix.'logo_url', $logo_url );
		update_post_meta( $post_id, 'wplss_slide_link', $logo_link );
	}

	/**
	 * Function to register admin menus
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_register_menu() {
		// Shortocde Mapper
		add_submenu_page( 'edit.php?post_type='.WPLS_PRO_POST_TYPE, __('Shortcode Generator', 'logoshowcase'), __('Shortcode Generator', 'logoshowcase'), 'edit_posts', 'wpls-shrt-mapper', array($this, 'wpls_pro_shortcode_mapper_page') );
		
		//Settings Page
		add_submenu_page( 'edit.php?post_type='.WPLS_PRO_POST_TYPE, __('Settings', 'logoshowcase'), __('Settings', 'logoshowcase'), 'manage_options', 'wpls-pro-settings', array($this, 'wpls_pro_settings_page') );
		
		// Shortocde Preview
		add_submenu_page( null, __('Shortcode Preview', 'logoshowcase'), __('Shortcode Preview', 'logoshowcase'), 'edit_posts', 'wpls-preview', array($this, 'wpls_pro_shortcode_preview_page') );
	}

	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.3.4
	 */
	function wpls_pro_shortcode_mapper_page() {
		include_once( WPLS_PRO_DIR . '/includes/admin/shortcode-mapper/wpls-shortcode-mapper.php' );
	}

	/**
	 * Function to handle the setting page html
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_settings_page() {
		include_once( WPLS_PRO_DIR . '/includes/admin/settings/wpls-settings.php' );
	}
	
	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.3.4
	 */
	function wpls_pro_shortcode_preview_page() {
	}

	/**
	 * Function to handle plugin shoercode preview
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.3.4
	 */
	function wpls_pro_generate_preview_screen( $screen ) {
		if( $screen->id == 'admin_page_wpls-preview' ) {
			include_once( WPLS_PRO_DIR . '/includes/admin/shortcode-mapper/shortcode-preview.php' );
			exit;
		}
	}

	/**
	 * Function register setings
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_admin_processes() {

		// If plugin notice is dismissed
		if( isset($_GET['message']) && $_GET['message'] == 'wpls-pro-plugin-notice' ) {
			set_transient( 'wpls_pro_install_notice', true, 604800 );
		}

		register_setting( 'wpls_pro_plugin_options', 'wpls_pro_options', array($this, 'wpls_pro_validate_options') );
	}

	/**
	 * Validate Settings Options
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_validate_options( $input ) {
		
		$input['tooltip_arrow']		= isset($input['tooltip_arrow']) 	? 1 : 0;
		$input['custom_css'] 		= isset($input['custom_css']) 		? wpls_pro_clean($input['custom_css'], true) 			: '';
		
		return $input;
	}

	/**
	 * Function to add category row action
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0
	 */
	function wpls_pro_add_tax_row_data( $actions, $tag ) {
		return array_merge( array( 'wpls_pro_id' => "<span style='color:#555'>ID: {$tag->term_id}</span>" ), $actions );
	}

	/**
	 * Add custom column to Logo listing page
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_posts_columns( $columns ){

		$new_columns['wpls_pro_image'] 	= __( 'Image', 'logoshowcase' );
		$new_columns['wpls_pro_order'] 	= __('Order', 'logoshowcase');

		$columns = wpls_pro_add_array( $columns, $new_columns, 1, true );

		return $columns;
	}

	/**
	 * Add custom column data to Logo listing page
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_post_columns_data( $column, $post_id ) {

		global $wp_query , $post;

		switch ($column) {

			case 'wpls_pro_image':
				$logo_image = wpls_pro_get_logo_image( $post_id, array(50, 50) );
				if( $logo_image ) {
					echo "<img src='{$logo_image}' class='wpls-logo-image' alt='' />";
				}
				break;

			case 'wpls_pro_order':

				$post_menu_order    = isset($post->menu_order) ? $post->menu_order : '';
			
				echo $post_menu_order;
				if ( isset( $wp_query->query['orderby'] ) && $wp_query->query['orderby'] == 'menu_order title' ) {
					echo "<input type='hidden' value='{$post_id}' name='wpls_pro_post[]' class='wpls-logo-order' id='wpls-logo-order-{$post_id}' />";
				}
				break;
		}
	}

	/**
	 * Add 'Sort Logo' link at Logo listing page
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_sorting_link( $views ) {
		
		global $post_type, $wp_query;

		$class            = ( isset( $wp_query->query['orderby'] ) && $wp_query->query['orderby'] == 'menu_order title' ) ? 'current' : '';
		$query_string     = remove_query_arg(array( 'orderby', 'order' ));
		$query_string     = add_query_arg( 'orderby', urlencode('menu_order title'), $query_string );
		$query_string     = add_query_arg( 'order', urlencode('ASC'), $query_string );
		$views['byorder'] = '<a href="' . esc_url( $query_string ) . '" class="' . esc_attr( $class ) . '">' . __( 'Sort Logo', 'logoshowcase' ) . '</a>';

		return $views;
	}

	/**
	 * Add category dropdown to Slider listing page
	 * 
	 * @package WP Logo Showcase Responsive Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_add_post_filters() {
		
		global $typenow;
		
		if( $typenow == WPLS_PRO_POST_TYPE ) {

			$wpls_pro_cat = isset($_GET[WPLS_PRO_CAT]) ? $_GET[WPLS_PRO_CAT] : '';

			$dropdown_options = apply_filters('wpls_pro_cat_filter_args', array(
					'show_option_all' 	=> __('All Categories', 'logoshowcase'),
					'hide_empty' 		=> 1,
					'hierarchical' 		=> 1,
					'show_count' 		=> 0,
					'orderby' 			=> 'name',
					'name'				=> WPLS_PRO_CAT,
					'taxonomy'			=> WPLS_PRO_CAT,
					'selected' 			=> $wpls_pro_cat,
					'value_field'		=> 'slug',
					'hide_if_empty'		=> true,
				));

			wp_dropdown_categories( $dropdown_options );
		}
	}

	/**
	 * Add Save button to Logo listing page
	 * 
	 * @package WP Logo Showcase and Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_restrict_manage_posts(){

		global $typenow, $wp_query;

		if( $typenow == WPLS_PRO_POST_TYPE && isset($wp_query->query['orderby']) && $wp_query->query['orderby'] == 'menu_order title' ) {

			$html  = '';
			$html .= "<span class='spinner wpls-spinner'></span>";
			$html .= "<input type='button' name='wpls_save_order' class='button button-secondary right wpls-save-order' id='wpls-save-order' value='".__('Save Sort Order', 'logoshowcase')."' />";
			echo $html;
		}
	}

	/**
	 * Update Blog order
	 * 
	 * @package WP Logo Showcase and Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_update_post_order() {

		// Taking some defaults
		$result 			= array();
		$result['success'] 	= 0;
		$result['msg'] 		= __('Sorry, Something happened wrong.', 'logoshowcase');

		if( !empty($_POST['form_data']) ) {

			$form_data 		= parse_str($_POST['form_data'], $output_arr);
			$wpls_posts 	= !empty($output_arr['wpls_pro_post']) ? $output_arr['wpls_pro_post'] : '';

			if( !empty($wpls_posts) ) {

				$post_menu_order = 0;

				// Loop od ids
				foreach ($wpls_posts as $wpls_post_key => $wpls_post) {
					
					// Update post order
					$update_post = array(
						'ID'           => $wpls_post,
						'menu_order'   => $post_menu_order,
					);

					// Update the post into the database
					wp_update_post( $update_post );

					$post_menu_order++;
				}

				$result['success'] 	= 1;
				$result['msg'] 		= __('Logo order saved successfully.', 'logoshowcase');
			}
		}
		echo json_encode($result);
		exit;
	}

	/**
	 * Function to add extra link to plugins action link
	 * 
	 * @package WP Logo Showcase and Slider Pro
	 * @since 1.0.0
	 */
	function wpls_pro_plugin_row_meta( $links, $file ) {
		
		if ( $file == WPLS_PRO_PLUGIN_BASENAME ) {
			
			$row_meta = array(
				'docs'    => '<a href="' . esc_url('http://docs.wponlinesupport.com/wp-logo-showcase-responsive-slider-pro/?utm_source=hp&event=doc') . '" title="' . esc_attr( __( 'View Documentation', 'logoshowcase' ) ) . '" target="_blank">' . __( 'Docs', 'logoshowcase' ) . '</a>',
				'support' => '<a href="' . esc_url('https://www.wponlinesupport.com/wordpress-support/?utm_source=hp&event=projobs') . '" title="' . esc_attr( __( 'Premium Support - For any Customization', 'logoshowcase' ) ) . '" target="_blank">' . __( 'Premium Support', 'logoshowcase' ) . '</a>',
			);
			return array_merge( $links, $row_meta );
		}
		return (array) $links;
	}
}

$wpls_pro_admin = new Wpls_Pro_Admin();